<?php
/**
 * Menu callback: Preset overview page.
 */
function presets_overview() {
  // Get the header for the overview page.
  $header = array(t('Preset Type'), t('Description'), array('data' => t('Operations'), 'colspan' => '3'));
  $rows = array();

  // Show each module that implements presets.
  foreach( module_implements('preset_info') as $module ) {
    $row = array();

    // Get the information.
    $info = preset_get_info( $module );

    // Check the permissions for this module.
    $permission = isset( $info['permission'] ) ? $info['permission']['name'] : 'administer presets';

    // Get the module admin path.
    $module_path = preset_get_module_path( $module );

    // See if the user has permissions to administer the presets for this module.
    if( user_access($permission) ) {
      $row[] = check_plain($info['title']);
      $row[] = array('data' => filter_xss_admin($info['description']), 'class' => 'description');
      $row[] = array('data' => l(t('Manage Presets'), $module_path));
      $row[] = array('data' => l(t('Add Preset'), $module_path . '/add'));
      $row[] = array('data' => l(t('Import Preset'), $module_path . '/import'));
      $rows[] = $row;
    }
  }

  if( count($rows) > 0 ) {
    return theme('table', array('header' => $header, 'rows' => $rows));
  }
  else {
    return t('No modules currently implementing presets, or you do not have <a href="/admin/people/permissions">permission</a>.');
  }
}

/**
 * Menu Callback:  Preset module overview.
 */
function preset_module_overview( $module ) {
  $presets = preset_get_presets( $module );
  if( count( $presets ) > 0 ) {
    $header = array(t('Preset Name'), t('Description'), array('data' => t('Operations'), 'colspan' => '3'));
    $rows = array();
    foreach( $presets as $pid => $preset ) {
      $row = array();

      // Get the preset admin path.
      $preset_path = preset_get_preset_path( $module, $preset['name'] );

      $row[] = check_plain($preset['name']);
      $row[] = array('data' => filter_xss_admin($preset['description']), 'class' => 'description');
      $row[] = array('data' => l(t('edit'), $preset_path));

      if( !$preset['default'] ) {
        $row[] = array('data' => l(t('delete'), $preset_path . '/delete'));
      }

      $row[] = array('data' => l(t('export'), $preset_path . '/export'));
      $rows[] = $row;
    }
    return theme('table', array('header' => $header, 'rows' => $rows));
  }
  else {
    return 'No presets.  To add a new preset, click on the <strong>Add Preset</strong> button.';
  }
}

function preset_array_flatten($array,$return=array()) {
  foreach( $array as $key => $value ) {
    if( is_array($value) ) {
      $return = preset_array_flatten($value, $return);
    }
    else {
      $return[$key] = $value;
    }
  }
  return $return;
}

/**
 * Form builder; the create preset form.
 */
function preset_create_form($form, &$form_state, $module) {

  // Include the necessary files.
  preset_include( $module );

  $form['preset'] = array(
    '#type' => 'fieldset',
    '#title' => t('Preset Information'),
    '#weight' => -10,
    '#tree' => true
  );

  $form['preset']['module'] = array(
    '#type' => 'value',
    '#value' => $module
  );

  $form['preset']['name'] = array(
    '#type' => 'textfield',
    '#title' => t('Preset Name'),
    '#maxlength' => PRESET_NAME_MAX_LENGTH,
    '#description' => t('Spaces are NOT allowed; punctuation is not allowed except for periods, hyphens, apostrophes, and underscores.'),
    '#required' => TRUE,
    '#weight' => -9,
  );

  $form['preset']['description'] = array(
    '#type' => 'textfield',
    '#title' => t('Description'),
    '#description' => t('Add a brief description to your preset name.'),
    '#weight' => -8,
  );

  if( $module_settings = module_invoke( $module, 'preset_create_form' ) ) {
    $form = array_merge_recursive( $form, $module_settings );
  }

  // Change the sumbit button to say "Next".
  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Next'),
    '#weight' => 30,
  );

  // Add the form submit handler.
  $form['#submit'][] = 'preset_create_submit';
  $form['#validate'][] = 'preset_create_validate';

  return $form;
}

/**
 * Validation for the preset form.
 */
function preset_create_validate($form, &$form_state) {
  $preset = $form_state['values']['preset'];
  if ($error = preset_validate_name($preset['module'], $preset['name'])) {
    form_set_error('name', $error);
  }
}

/**
 * Submit handler for the preset form.
 */
function preset_create_submit($form, &$form_state) {
  $preset = $form_state['values']['preset'];
  $preset['name'] = strtolower( $preset['name'] );

  // Only store the unique settings...
  $preset['settings'] = preset_array_flatten($preset['settings']);
  $preset['settings'] = preset_get_unique_settings( $preset );

  preset_set($preset);
  $form_state['redirect'] = preset_get_preset_path( $preset['module'], $preset['name'] );
}

function preset_form( $form, &$form_state, $preset ) {

  // Include the necessary files.
  preset_include( $preset['module'] );
  
  $form['preset'] = array(
    '#type' => 'fieldset',
    '#title' => t('Preset Settings'),
    '#collapsible' => false,
    '#collapsed' => false,
    '#tree' => true
  );

  $form['preset'] = array_merge( $form['preset'], array(
    'pid' => array('#type' => 'value', '#value' => $preset['pid']),
    'module' => array('#type' => 'value', '#value' => $preset['module']),
    'name' => array('#type' => 'value', '#value' => $preset['name'])
  ));

  $form['preset']['description'] = array(
    '#type' => 'textfield',
    '#title' => t('Description'),
    '#description' => t('Add a brief description to your preset name.'),
    '#default_value' => $preset['description'],
    '#weight' => -9,
  );

  if( $preset_form = module_invoke( $preset['module'], 'preset_form', $preset ) ) {
    // Now add the module specific settings form to this fieldset.
    $form = array_merge_recursive( $form, $preset_form );
  }

  $form['settings'] = array(
    '#type' => 'value',
    '#value' => $preset['settings']
  );

  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Save Preset'),
    '#weight' => 30,
  );

  $form['delete'] = array(
    '#type' => 'submit',
    '#value' => t('Delete'),
    '#weight' => 40,
    '#submit' => array('preset_delete_submit'),
  );

  // Add the form submit handler.
  $form['#submit'][] = 'preset_submit';
  $form['#validate'][] = 'preset_validate';
  return $form;
}

/**
 * Validation for the preset form.
 */
function preset_validate($form, &$form_state) {
}

/**
 * Submit handler for the preset form.
 */
function preset_submit($form, &$form_state) {
  $preset = array();
  $preset = $form_state['values']['preset'];

  // Make sure we include all settings with the ones from the form overriding the defaults.
  $preset['settings'] = array_merge( 
    preset_array_flatten( $form_state['values']['settings'] ),
    preset_array_flatten( $preset['settings'] )
  );

  // Only store the unique settings...
  $preset['settings'] = preset_get_unique_settings( $preset );

  // Set the value of this preset.
  preset_set( $preset );
  $form_state['redirect'] = preset_get_preset_path( $preset['module'], $preset['name'] );
}

/**
 * Button sumit function: handle the 'Delete' button on the preset form.
 */
function preset_delete_submit($form, &$form_state) {
  $preset = $form_state['values']['preset'];

  // Get the preset admin path.
  $preset_path = preset_get_preset_path( $preset['module'], $preset['name'] );

  // Redirect to the delete path.
  $form_state['redirect'] = $preset_path . '/delete';
}

/**
 * Menu callback; delete a single preset.
 */
function preset_delete_confirm($form, &$form_state, $preset) {
  $form['preset'] = array(
    '#type' => 'value',
    '#value' => $preset 
  );
  $message = t('Are you sure you want to delete the preset %name?', array('%name' => $preset['name']));
  $caption = '<p>' . t('This action cannot be undone.') . '</p>';
  return confirm_form($form, $message, preset_get_module_path( $preset['module'] ), $caption, t('Delete'));
}

/**
 * Execute preset deletion
 */
function preset_delete_confirm_submit($form, &$form_state) {
  $preset = $form_state['values']['preset'];
  if( $form_state['values']['confirm'] ) {
    preset_delete($preset['module'], $preset['name']);
  }
  $form_state['redirect'] = preset_get_module_path( $preset['module'] );
}

/**
 * Import's a preset.
 */
function preset_import_form( $form, &$form_state ) {
  $form['preset'] = array(
    '#title' => t('Preset Code'),
    '#type' => 'textarea',
    '#rows' => 10,
    '#description' => t('Copy the export text and paste it into this text field to import a new preset.'),
  );

  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Import Preset'),
    '#weight' => 30,
  );

  $form['#submit'][] = 'preset_import_submit';
  $form['#validate'][] = 'preset_import_validate';

  return $form;
}

/**
 * Validation for the preset import form.
 */
function preset_import_validate($form, &$form_state) {
  $preset = '';

  // Get the preset that they declared in the text field.
  ob_start();
  eval($form_state['values']['preset']);
  ob_end_clean();

  if( is_array($preset) ) {
    $module = isset( $preset['module'] ) ? $preset['module'] : '';
    $name = isset( $preset['name'] ) ? $preset['name'] : '';
    if ($error = preset_validate_name($module, $name)) {
      form_set_error('name', $error);
    }
  }
  else {
    form_set_error('name', 'Invalid preset import.');
  }

  $form_state['preset'] = &$preset;
}

/**
 * Submit handler for the preset import form.
 */
function preset_import_submit($form, &$form_state) {
  $preset = $form_state['preset'];
  preset_set( $preset );
  $form_state['redirect'] = preset_get_preset_path( $preset['module'], $preset['name'] );
}

/**
 * The preset export form.
 */
function preset_export_form( $form, &$form_state, $preset ) {
  // Unset the unnecessary elements.
  unset($preset['pid']);
  unset($preset['default']);

  // Get the code string representation.
  $code = var_export($preset, true);

  // Make sure to format the arrays like drupal.
  $code = str_replace("=> \n  ", '=> ', $code);

  // Add the preset variable.
  $code = '$preset = ' . $code . ';';

  $lines = substr_count($code, "\n");
  $form['export'] = array(
    '#title' => t('Export Preset'),
    '#type' => 'textarea',
    '#value' => $code,
    '#rows' => $lines,
    '#description' => t('Copy the export text and paste it into another preset using the import function.'),
  );

  // Return the form.
  return $form;
}
